from datetime import timedelta
from fastapi import APIRouter, Depends, HTTPException, Response, status, Request
from sqlalchemy.orm import Session
from app.core import deps
from app.schemas import auth as auth_schemas
from app.schemas import user as user_schemas
from app.services import auth as auth_service
from app.models import User
from app.models.club import Club

router = APIRouter()


@router.get("/me", response_model=user_schemas.UserRead)
def get_current_user_info(current_user: User = Depends(deps.get_current_user)):
    """Get current authenticated user's information"""
    return current_user


@router.post("/login", response_model=auth_schemas.Token)
def login(data: auth_schemas.LoginRequest, response: Response, db: Session = Depends(deps.get_db)):
    user = auth_service.authenticate_user(db, data.email, data.password)
    if not user:
        raise HTTPException(status_code=status.HTTP_401_UNAUTHORIZED, detail="Invalid credentials")

    access_token = auth_service.create_access_token_for_user(user)
    refresh_token = auth_service.create_refresh_token_for_user(user)

    # set refresh token as HttpOnly secure cookie
    response.set_cookie(
        key="refresh_token",
        value=refresh_token,
        httponly=True,
        secure=False,  # set True in production with HTTPS
        samesite="lax",
        max_age=auth_service.REFRESH_TOKEN_EXPIRE_SECONDS,
    )

    return {"access_token": access_token, "token_type": "bearer", "role": user.role}


@router.post("/refresh", response_model=auth_schemas.Token)
def refresh(request: Request, response: Response, db: Session = Depends(deps.get_db)):
    token = request.cookies.get("refresh_token")
    if not token:
        raise HTTPException(status_code=status.HTTP_401_UNAUTHORIZED, detail="Missing refresh token")
    user = auth_service.verify_refresh_token(db, token)
    if not user:
        raise HTTPException(status_code=status.HTTP_401_UNAUTHORIZED, detail="Invalid refresh token")

    access_token = auth_service.create_access_token_for_user(user)
    refresh_token = auth_service.create_refresh_token_for_user(user)
    response.set_cookie(
        key="refresh_token",
        value=refresh_token,
        httponly=True,
        secure=False,
        samesite="lax",
        max_age=auth_service.REFRESH_TOKEN_EXPIRE_SECONDS,
    )
    return {"access_token": access_token, "token_type": "bearer", "role": user.role}


@router.post("/logout")
def logout(response: Response):
    response.delete_cookie("refresh_token")
    return {"msg": "logged out"}


@router.post("/trainer-invite")
def create_trainer_invite(
    request: Request,
    club_id: int | None = None,
    current_admin: User = Depends(deps.get_current_admin),
    db: Session = Depends(deps.get_db)
):
    """Create a secure, time-limited invite token for trainer registration"""
    # If club_id provided, bind invite to that club
    if club_id:
        # Verify club exists and belongs to this admin
        club = db.query(auth_service.Club).filter(Club.id == club_id, Club.admin_id == current_admin.id).first()
        if not club:
            raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Club not found")
        token = auth_service.create_trainer_invite_token(db, current_admin.id, club_id=club_id)
    else:
        token = auth_service.create_trainer_invite_token(db, current_admin.id)
    # Don't build the full URL - let frontend handle it
    # Frontend will be on a different port than backend
    return {"token": token, "expires_in_hours": 24}


@router.get("/trainer-invites")
def list_trainer_invites(
    current_admin: User = Depends(deps.get_current_admin),
    db: Session = Depends(deps.get_db)
):
    """List all active trainer invites created by current admin"""
    invites = auth_service.list_admin_trainer_invites(db, current_admin.id)
    return invites


@router.delete("/trainer-invite/{token}")
def revoke_trainer_invite(
    token: str,
    current_admin: User = Depends(deps.get_current_admin),
    db: Session = Depends(deps.get_db)
):
    """Revoke a trainer invite token"""
    success = auth_service.revoke_trainer_invite(db, token, current_admin.id)
    if not success:
        raise HTTPException(status_code=404, detail="Invite not found or already used")
    return {"message": "Invite revoked successfully"}


@router.post("/parent-invite")
def create_parent_invite(
    current_user: User = Depends(deps.get_current_trainer),
    db: Session = Depends(deps.get_db)
):
    """Create a secure, time-limited invite token for parent registration (trainer only)"""
    if not current_user.club_id:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail="Trainer not assigned to a club"
        )
    
    token = auth_service.create_parent_invite(db, current_user.id, current_user.club_id)
    return {"token": token, "club_id": current_user.club_id}


@router.get("/parent-invites")
def list_parent_invites(
    current_user: User = Depends(deps.get_current_trainer),
    db: Session = Depends(deps.get_db)
):
    """List all active parent invites created by current trainer"""
    invites = auth_service.list_trainer_parent_invites(db, current_user.id)
    return invites


@router.delete("/parent-invite/{token}")
def revoke_parent_invite(
    token: str,
    current_user: User = Depends(deps.get_current_trainer),
    db: Session = Depends(deps.get_db)
):
    """Revoke a parent invite token"""
    success = auth_service.revoke_parent_invite(db, token, current_user.id)
    if not success:
        raise HTTPException(status_code=404, detail="Invite not found or already used")
    return {"message": "Invite revoked successfully"}


@router.get("/validate-invite/{token}")
def validate_invite(
    token: str,
    db: Session = Depends(deps.get_db)
):
    """Validate an invite token and return its details"""
    try:
        # Try parent invite first
        invite = auth_service.validate_parent_invite_token(db, token)
        club = db.query(Club).get(invite.club_id)
        return {
            "valid": True,
            "role": "PARENT",
            "club_id": invite.club_id,
            "club_name": club.name if club else None,
            "expires_at": invite.expires_at.isoformat()
        }
    except:
        try:
            # Try trainer invite
            invite = auth_service.validate_trainer_invite_token(db, token)
            # If trainer invite has club_id, return club details
            club_name = None
            city = None
            if invite.club_id:
                club = db.query(Club).get(invite.club_id)
                if club:
                    club_name = club.name
                    city = club.city
            return {
                "valid": True,
                "role": "TRAINER",
                "club_id": invite.club_id,
                "club_name": club_name,
                "city": city,
                "expires_at": invite.expires_at.isoformat()
            }
        except Exception as e:
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail=str(e)
            )


@router.post("/register/trainer")
def register_trainer(payload: auth_schemas.TrainerRegister, db: Session = Depends(deps.get_db)):
    try:
        user = auth_service.register_trainer(db, payload)
        return {"id": user.id, "email": user.email}
    except ValueError as e:
        raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail=str(e))


@router.post("/register/parent")
def register_parent(payload: auth_schemas.ParentRegister, db: Session = Depends(deps.get_db)):
    try:
        user = auth_service.register_parent(db, payload)
        return {"id": user.id, "email": user.email}
    except ValueError as e:
        raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail=str(e))


@router.post('/forgot-password')
def forgot_password(payload: auth_schemas.ForgotRequest, db: Session = Depends(deps.get_db)):
    user = db.query(deps.SessionLocal().bind.mapper_registry._class_registry.get('User')).filter_by(email=payload.email).first() if False else None
    # simple lookup
    user = db.query(deps.get_db().__self__.__func__.__globals__['User']).filter_by(email=payload.email).first() if False else None
    # fallback: query directly from models
    try:
        from app.models.user import User as UserModel
        user = db.query(UserModel).filter(UserModel.email == payload.email).first()
    except Exception:
        user = None

    if not user:
        # don't reveal whether email exists
        return {"msg": "If that email exists, a reset link was sent"}

    token = auth_service.create_password_reset_token(user)
    # TODO: send token via email. For now return token in response for dev.
    # return {"msg": "reset token (dev)", "token": token}
    
    # delegate to service which will create token and send email if user exists
    auth_service.send_password_reset_email(db, payload.email)
    # Always return neutral response to avoid account enumeration
    return {"msg": "If that email exists, you will receive password reset instructions."}

@router.post('/reset-password')
def reset_password(payload: auth_schemas.ResetRequest, db: Session = Depends(deps.get_db)):
    user = auth_service.reset_password(db, payload.token, payload.password)
    if not user:
        raise HTTPException(status_code=status.HTTP_400_BAD_REQUEST, detail="Invalid or expired token")
    return {"msg": "Password reset successful"}
