from fastapi import APIRouter, Depends, HTTPException, status
from sqlalchemy.orm import Session
from typing import List
from app.core import deps
from app.models import Club, User, RoleEnum
from app.schemas import club as club_schemas

router = APIRouter()


@router.get("/", response_model=List[club_schemas.ClubRead])
def list_clubs(
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_user),
    skip: int = 0,
    limit: int = 100
):
    """List all clubs - admins only see clubs they manage, includes trainer info"""
    query = db.query(Club)
    
    # If user is admin (COMPANY_EMPLOYEE), filter by clubs they manage
    if current_user.role == RoleEnum.COMPANY_EMPLOYEE:
        query = query.filter(Club.admin_id == current_user.id)
    
    clubs = query.offset(skip).limit(limit).all()
    
    # Enrich with trainer information
    result = []
    for club in clubs:
        club_dict = {
            "id": club.id,
            "name": club.name,
            "city": club.city,
            "admin_id": club.admin_id,
            "trainer_name": None,
            "trainer_email": None
        }
        
        # Find trainer for this club
        trainer = db.query(User).filter(
            User.club_id == club.id,
            User.role == RoleEnum.TRAINER
        ).first()
        
        if trainer:
            trainer_name_parts = []
            if trainer.first_name:
                trainer_name_parts.append(trainer.first_name)
            if trainer.last_name:
                trainer_name_parts.append(trainer.last_name)
            club_dict["trainer_name"] = " ".join(trainer_name_parts) if trainer_name_parts else trainer.email
            club_dict["trainer_email"] = trainer.email
        
        result.append(club_dict)
    
    return result


@router.get("/{club_id}", response_model=club_schemas.ClubRead)
def get_club(
    club_id: int,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_user)
):
    """Get a specific club by ID with trainer information"""
    club = db.query(Club).filter(Club.id == club_id).first()
    if not club:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Club not found")
    
    # Enrich with trainer information
    club_dict = {
        "id": club.id,
        "name": club.name,
        "city": club.city,
        "admin_id": club.admin_id,
        "trainer_name": None,
        "trainer_email": None
    }
    
    # Find trainer for this club
    trainer = db.query(User).filter(
        User.club_id == club.id,
        User.role == RoleEnum.TRAINER
    ).first()
    
    if trainer:
        trainer_name_parts = []
        if trainer.first_name:
            trainer_name_parts.append(trainer.first_name)
        if trainer.last_name:
            trainer_name_parts.append(trainer.last_name)
        club_dict["trainer_name"] = " ".join(trainer_name_parts) if trainer_name_parts else trainer.email
        club_dict["trainer_email"] = trainer.email
    
    return club_dict


@router.post("/", response_model=club_schemas.ClubRead, status_code=status.HTTP_201_CREATED)
def create_club(
    club_in: club_schemas.ClubCreate,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_admin)
):
    """Create a new club (admin only) - automatically assigns the creating admin as the club manager"""
    club_data = club_in.model_dump()
    # Automatically assign the creating admin to manage this club
    club_data['admin_id'] = current_user.id
    club = Club(**club_data)
    db.add(club)
    db.commit()
    db.refresh(club)
    return club


@router.put("/{club_id}", response_model=club_schemas.ClubRead)
def update_club(
    club_id: int,
    club_in: club_schemas.ClubUpdate,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_admin)
):
    """Update a club (admin only)"""
    club = db.query(Club).filter(Club.id == club_id).first()
    if not club:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Club not found")
    
    update_data = club_in.model_dump(exclude_unset=True)
    for field, value in update_data.items():
        setattr(club, field, value)
    
    db.commit()
    db.refresh(club)
    return club


@router.delete("/{club_id}", status_code=status.HTTP_204_NO_CONTENT)
def delete_club(
    club_id: int,
    db: Session = Depends(deps.get_db),
    current_user: User = Depends(deps.get_current_admin)
):
    """Delete a club (admin only)"""
    club = db.query(Club).filter(Club.id == club_id).first()
    if not club:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Club not found")
    
    db.delete(club)
    db.commit()
    return None

