from pydantic import BaseModel, field_validator
from typing import List, Optional
from decimal import Decimal
from datetime import datetime


class BasketItemBase(BaseModel):
    product_id: int
    variant_id: int
    quantity: int
    customization_text: Optional[str] = None
    
    @field_validator('quantity')
    @classmethod
    def validate_quantity(cls, v):
        if v < 1:
            raise ValueError('Quantity must be at least 1')
        return v


class BasketItemCreate(BasketItemBase):
    pass


class BasketItemUpdate(BaseModel):
    quantity: int
    
    @field_validator('quantity')
    @classmethod
    def validate_quantity(cls, v):
        if v < 1:
            raise ValueError('Quantity must be at least 1')
        return v


class BasketItemRead(BasketItemBase):
    id: int
    basket_id: int
    product_name: str
    variant_size: str
    unit_price: Decimal
    total_price: Decimal

    class Config:
        from_attributes = True


class BasketBase(BaseModel):
    child_id: int
    club_id: int


class BasketCreate(BasketBase):
    pass


class BasketRead(BasketBase):
    id: int
    parent_id: int
    created_at: datetime
    updated_at: datetime
    items: List[BasketItemRead]
    total_amount: Decimal

    class Config:
        from_attributes = True


class BasketSummary(BaseModel):
    """Summary of basket for quick display"""
    id: int
    child_id: int
    club_id: int
    item_count: int
    total_amount: Decimal

    class Config:
        from_attributes = True
