"""
File storage utility for handling image uploads.
Designed to be easily swappable with cloud storage later.
"""
import os
import uuid
from pathlib import Path
from typing import Optional
from fastapi import UploadFile
import shutil

# Configuration - easy to change for cloud storage later
UPLOAD_DIR = Path("uploads/products")
MAX_FILE_SIZE = 5 * 1024 * 1024  # 5MB
ALLOWED_EXTENSIONS = {".jpg", ".jpeg", ".png", ".gif", ".webp"}


def ensure_upload_dir():
    """Ensure upload directory exists"""
    UPLOAD_DIR.mkdir(parents=True, exist_ok=True)


def validate_image(file: UploadFile) -> tuple[bool, Optional[str]]:
    """
    Validate uploaded image file
    Returns: (is_valid, error_message)
    """
    # Check file extension
    if file.filename:
        ext = Path(file.filename).suffix.lower()
        if ext not in ALLOWED_EXTENSIONS:
            return False, f"Invalid file type. Allowed: {', '.join(ALLOWED_EXTENSIONS)}"
    
    return True, None


async def save_product_image(file: UploadFile) -> str:
    """
    Save uploaded product image to local storage.
    Returns: relative path to the saved file (e.g., "uploads/products/uuid_filename.jpg")
    
    This function can be replaced with cloud storage upload later.
    """
    ensure_upload_dir()
    
    # Validate file
    is_valid, error = validate_image(file)
    if not is_valid:
        raise ValueError(error)
    
    # Generate unique filename
    file_ext = Path(file.filename).suffix.lower() if file.filename else ".jpg"
    unique_filename = f"{uuid.uuid4()}{file_ext}"
    file_path = UPLOAD_DIR / unique_filename
    
    # Save file
    try:
        with open(file_path, "wb") as buffer:
            shutil.copyfileobj(file.file, buffer)
    finally:
        await file.close()
    
    # Return relative path that can be used in URLs
    return f"uploads/products/{unique_filename}"


def delete_product_image(image_path: str) -> bool:
    """
    Delete product image from storage.
    Returns: True if deleted, False if not found
    
    This function can be replaced with cloud storage deletion later.
    """
    if not image_path:
        return False
    
    try:
        file_path = Path(image_path)
        if file_path.exists():
            file_path.unlink()
            return True
    except Exception as e:
        print(f"Error deleting image {image_path}: {e}")
    
    return False


def get_image_url(image_path: Optional[str], base_url: str = "") -> Optional[str]:
    """
    Convert image path to full URL.
    For local storage: http://localhost:8000/uploads/products/file.jpg
    For cloud storage: https://cdn.example.com/products/file.jpg
    
    Args:
        image_path: Relative path to image (e.g., "uploads/products/file.jpg")
        base_url: Base URL of the server (e.g., "http://localhost:8000")
    
    Returns: Full URL to the image
    """
    if not image_path:
        return None
    
    # For local storage
    if not image_path.startswith("http"):
        return f"{base_url}/{image_path}"
    
    # Already a full URL (from cloud storage)
    return image_path
